/*******************************************************************************
 *
 *      G R A P H I C S    P R I M I T I V E S
 *
 *      Copyright A Levido 2013 - All Rights Reserved
 *
 ******************************************************************************/
#ifndef GRDRIVER_H
#define GRDRIVER_H
#include "genericTypeDefs.h"
#include "display.h"

/* Colours */
#define RGBConvert(red, green, blue)(UINT16)((((UINT16)(red) & 0xF8) << 8) \
    | (((UINT16)(green) & 0xFC) << 3) | ((UINT16)(blue) >> 3))


#define WHITE           RGBConvert(0xff, 0xff, 0xff)
#define BLACK           RGBConvert(0x00, 0x00, 0x00)

#define RED             RGBConvert(0xff, 0x00, 0x00)
#define BLUE            RGBConvert(0x00, 0x00, 0xff) // good

#define GREEN           RGBConvert(0x00, 0xcc, 0x00)

#define YELLOW          RGBConvert(0xcc, 0xcc, 0x00)
#define PURPLE          RGBConvert(0xdd, 0x00, 0xcc)
#define CYAN            RGBConvert(0x00, 0xee, 0xcc)

#define LT_GREY         RGBConvert(0xee, 0xee, 0xee)
#define LTLT_GREY       RGBConvert(0xf7, 0xfb, 0xf7)


#define DK_BLUE         RGBConvert(0x00, 0x33, 0xcc)
#define DK_RED          RGBConvert(0xdd, 0x00, 0x00)
#define ORANGE          RGBConvert(0xff, 0xcc, 0x00)
#define BROWN           RGBConvert(0xcc, 0xcc, 0x33)
#define OLIVE           RGBConvert(0x66, 0xcc, 0x33)
#define AQUA            RGBConvert(0x33, 0x99, 0xcc)

// Quadrants
#define QUADRANT_1          0x01 // top right
#define QUADRANT_2          0x02 // top left
#define QUADRANT_3          0x04 // bottom left
#define QUADRANT_4          0x08 // bottom right
#define QUADRANT_A          0x0f // all for quadrants

/* Fonts */
extern const UINT16 const* font12[];
extern const UINT16 const* font16[];
#define FONT const UINT16 const*

/* Icons */
extern const UINT8 iconPlay[];
extern const UINT8 iconStop[];
extern const UINT8 iconRecord[];
extern const UINT8 iconPause[];
extern const UINT8 iconSkipFwd[];
extern const UINT8 iconSkipBack[];
extern const UINT8 iconVolUp[];
extern const UINT8 iconVolDn[];
extern const UINT8 iconSpkMute[];
extern const UINT8 iconSpkUnmute[];
extern const UINT8 iconMic[];
extern const UINT8 iconMicMute[];
extern const UINT8 iconLineIn[];
extern const UINT8 iconConfig[];
extern const UINT8 iconNext[];
extern const UINT8 iconPrev[];
extern const UINT8 iconPlus[];
extern const UINT8 iconMinus[];
extern const UINT8 iconLineUp[];
extern const UINT8 iconLineDn[];
extern const UINT8 iconMicBoost[];
extern const UINT8 iconClock[];
extern const UINT8 iconCal[];
extern const UINT8 iconUSBon[];
extern const UINT8 iconUSBoff[];
extern const UINT8 iconUp[];
extern const UINT8 iconDown[];
extern const UINT8 iconSort[];
extern const UINT8 iconNoSort[];
extern const UINT8 iconPlaySingle[];
extern const UINT8 iconPlayMulti[];

#define THIN        0
#define THICK       1

/* macros for lcd Functions*/
#define grInit(bgColour) lcdInit(bgColour);
#define grBlock(x1, y1, x2, y2, colour) lcdBlock(x1, y1, x2, y2, colour)
#define grSetBacklight(level) lcdSetBacklight(level)

#define grIcon(x, y, fg, bg, ic) lcdIcon(x, y, fg, bg, ic)
void lcdRender(INT16 x1, INT16 y1, UINT16 width, UINT16 height, UINT16 fg, UINT16 bg, const UINT16 * pglyph);

/** Draw Line
 *  @param x1       - leftmost pixel
 *  @param y1       - topmost pixel
 *  @param x2       - rightmost pixel
 *  @param y2       - bottommost pixel
 *  @param colour   - colour of block in 5:6:5 RGB format
 *  @param thick    - FALSE = 1 pixel wide, TRUE = 3 pixels wide
 */
void grLine(INT16 x1, INT16 y1, INT16 x2, INT16 y2, UINT16 colour, BOOL thick);

/** Draw Box
 *  @param x1       - leftmostpixel
 *  @param y1       - topmost pixel
 *  @param x2       - rightmost pixel
 *  @param y2       - bottommostpixel
 *  @param colour   - colour of box in 5:6:5 RGB format
 *  @param thick    - FALSE = 1 pixel wide, TRUE = 3 pixels wide
 */
void grBox(INT16 x1, INT16 y1, INT16 x2, INT16 y2, UINT16 colour, BOOL thick);

/** Draw Text
 *  @param x1       - location x pixel
 *  @param y1       - location y pixel
 *  @param col      - colour of text in 5:6:5 RGB format
 *  @param bg       - colour of background in 5:6:5 RGB format
 *  @param f        - pointer to Microchip font
 *  @param centred  - FALSE x1,y1 = top left, TRUE x1,y1 = centre
 */
void grText(UINT16 x1, UINT16 y1, UINT16 col, UINT16 bg, const UINT16 const* f[], BOOL centred, char* txt);

UINT16 grGetHeight(FONT font[]);

UINT16 grGetWidth(FONT font[], char* text);

UINT16 grFit(FONT font[], char* text, INT16 width);


/** Draw Icon
 *  @param x1       - location x pixel
 *  @param y1       - location y pixel
 *  @param col      - colour of icon in 5:6:5 RGB format
 *  @param bg       - colour of background in 5:6:5 RGB format
 *  @param icon[]     - pointer to 24 x 24 bit icon 3 byte x 24 array
 */
//void grIcon(UINT16 x1, UINT16 y1, UINT16 col, UINT16 bg, const UINT8 icon[]);



/** Draw one or more quadrants of a circle
 *
 * Draws one or more quadrants in the curent color line thickness & type
 * centred on a point (x,y) and with radius r.
 *
 *            |             QUADRANT_1
 *        II  |  I          QUADRANT_2
 *            |             QUADRANT_3
 *      ------+------       QUADRANT_4
 *            |             QUADRANT_A (all 4 quadrants)
 *        III | IV
 *            |
 *
 * @param x x-coordinate of the centre of the circle
 * @param y y-coordinate of the centre of the circle
 * @param r radius of the circle
 * @param quadrants Quadrant flags
 */
void grQuadrants (INT16 x0, INT16 y0, INT16 r, UINT8 quadrants, UINT16 colour, BOOL thick);

/** Draw a circle
 *
 * Draws a circle in the curent color line thickness & type
 * centred on a point (x,y) and with radius r.
 *
 * @param x x-coordinate of the centre of the circle
 * @param y y-coordinate of the centre of the circle
 * @param r radius of the circle
 */
#define grCircle(x, y, r, c, t) grQuadrants((x), (y), (r), QUADRANT_A, (c), (t));

/** Draw a rounded rectangle
 *
 * Draws a rounded rectangle in the curent color line thickness & type
 * with with corner radius r.
 *
 *      *------------*
 *    *  +(x0,y0)    + *                      '
 *    |                |
 *    *  +    (x1,y1)+ *
 *      *------------*
 *
 * @param x0 x-coordinate of top left point
 * @param y0 y-coordinate of top left point
 * @param x1 x-coordinate of bottom right point
 * @param y1 y-coordinate of bottom right point
 */
void grRoundRect (INT16 x0, INT16 y0, INT16 x1, INT16 y1, INT16 r, UINT16 colour, BOOL thick);


/** Draw one or more filled quadrants of a circle
 *
 * Draws one or more filled quadrants in the curent color
 * centred on a point (x,y) and with radius r.
 *
 *            |             QUADRANT_1
 *        II  |  I          QUADRANT_2
 *            |             QUADRANT_3
 *      ------+------       QUADRANT_4
 *            |             QUADRANT_A (all 4 quadrants)
 *        III | IV
 *            |
 *
 * @param x x-coordinate of the centre of the circle
 * @param y y-coordinate of the centre of the circle
 * @param r radius of the circle
 * @param quadrants Quadrant flags
 */
void grFilledQuadrants (INT16 x0, INT16 y0, INT16 r, UINT8 quadrants, UINT16 colour);

/** Draw a filled circle
 *
 * Draws a filed circle in the curent color
 * centred on a point (x,y) and with radius r.
 *
 * @param x x-coordinate of the centre of the circle
 * @param y y-coordinate of the centre of the circle
 * @param r radius of the circle
 */
#define grFilledCircle(x, y, r, c) grFilledQuadrants((x), (y), (r), QUADRANT_A, (c))

/** Draw a filled rounded rectangle
 *
 * Draws a filled rounded rectangle in the curent color
 * with with corner radius r.
 *
 *    + *------------*
 *    *  (x0,y0)       *                      '
 *    |                |
 *    *       (x1,y1)  *
 *      *------------* +
 *
 * @param x0 x-coordinate of top left point
 * @param y0 y-coordinate of top left point
 * @param x1 x-coordinate of bottom right point
 * @param y1 y-coordinate of bottom right point
 */
void grFilledRoundRect(SHORT x1, SHORT y1, SHORT x2, SHORT y2, SHORT rad, UINT16 colour);

#endif